<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

class ApplicationModel extends CI_Model {
	
    function __construct() {
        parent::__construct();
		$this->load->database();
    }
    
    /*
     * get all application data by application id
     */
    public function get_application($id){
    	return $this->db->query('SELECT * FROM application WHERE id = '. $this->db->escape($id));
    }
    
    /*
     * get a list of applications according to your access level, if app_access is null then it is assumed you are admin
     */
    public function get_applications($app_access = NULL){    	
    	if ($app_access === NULL) {
    		return $this->db->query('SELECT * FROM application ORDER BY LOWER(name)');
		}
    	else {
			if(is_array($app_access)) {
				foreach($app_access as $app) { if(!is_numeric($app)) { return FALSE; } }
				$app_access_str = implode(',',$app_access);
				return $this->db->query('SELECT * FROM application where id in ('.$app_access_str.') ORDER BY LOWER(name)');
			}
			return FALSE;
		}
    }
    
    public function get_applications_by_page_number($app_access = NULL, $start, $size){
    	if ($app_access === NULL) {
    		return $this->db->query('SELECT * FROM (select *, ROW_NUMBER() over (order by LOWER(name)) as row FROM application) a where row >= ' . $this->db->escape($start) . ' and row < ' . $this->db->escape($size+$start));
    	}
    	else {
    		//validate input by checking that each item in the array is numeric
    		if(is_array($app_access)) {
	    		foreach($app_access as $app) { if(!is_numeric($app)) { return FALSE; } }
	    		$app_access_str = implode(',',$app_access);
	    		return $this->db->query('SELECT * FROM (select *, ROW_NUMBER() over (order by LOWER(name)) as row FROM application) a where id in ('.$app_access_str.') and row >= ' . $this->db->escape($start) . ' and row < ' . $this->db->escape($size+$start));
    		}
    		return FALSE;
    	}
    }
    
	public function get_app_list_size($app_access = NULL)//get size of entire list
	{
		if ($app_access === NULL) {
			return $this->db->query('select COUNT(*) as count from application');
		}
		else {
			//validate input by checking that each item in the array is numeric
			foreach($app_access as $app) { if(!is_numeric($app)) { return FALSE; } }
			$app_access_str = implode(',',$app_access);
			return $this->db->query('select COUNT(*) as count from application where id in ('.$app_access_str.')');
		}
	}
    
    /*
     * get application information from the application's public key
     */
    public function get_application_from_public($public){
    	return $this->db->query('SELECT * FROM application WHERE public_key = '.$this->db->escape($public));
    }
    
    /*
     * get application information from the application's private key
    */
    public function get_application_from_private($private){
    	return $this->db->query('SELECT * FROM application WHERE private_key = '.$this->db->escape($private));
    }
	
    /*
     * get just the application id from the public key
     */
    public function get_application_id_from_public($public){
    	$query = $this->db->query('SELECT id FROM application WHERE public_key = '.$this->db->escape($public));
		if($query) {
			if($query->num_rows === 1) {
				$row = $query->row_array();
				return $row['id'];
			}
			else { return FALSE; }
		}
		else { return FALSE; }
    }
	
	/* This function tests if an application is a member of the LDAP group that gives permission
	 * to use the Direct API functions. It returns TRUE if the application has access and FALSE on
	 * failure to retrieve permissions or if the application does not have access.
	 */
	public function get_direct_api_authorization($id) {
		$app_dn = $this->get_dn_from_app_id($id);
		if($app_dn) {
			$ldap_conn = $this->prepare_ldap_conn();
			$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
			if($ldap_bind) {
				$dn_exp = ldap_explode_dn(LDAP_DIRECT_API_PERMISSIONS_GROUP,1);
				$search = ldap_search($ldap_conn, LDAP_BASE_RDN, '(&(ou='.$dn_exp[0].'))', array('member'));
				$entries = ldap_get_entries($ldap_conn, $search);
				//grab the member entries
				$apps = $entries[0]['member'];
				foreach($apps as $dn) {
					if($dn === $app_dn) { return TRUE; }
				}
			}
		}
		return FALSE;
	}
	
	/* This function tests if an application is a member of the LDAP group that gives permission
	 * to use the Admin API functions. It returns TRUE if the application has access and FALSE on
	 * failure to retrieve permissions or if the application does not have access.
	 */
	public function get_admin_api_authorization($id) {
		$app_dn = $this->get_dn_from_app_id($id);
		if($app_dn) {
			$ldap_conn = $this->prepare_ldap_conn();
			$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
			if($ldap_bind) {
				$dn_exp = ldap_explode_dn(LDAP_ADMIN_API_PERMISSIONS_GROUP,1);
				$search = ldap_search($ldap_conn, LDAP_BASE_RDN, '(&(ou='.$dn_exp[0].'))', array('member'));
				$entries = ldap_get_entries($ldap_conn, $search);
				//grab the member entries
				$apps = $entries[0]['member'];
				foreach($apps as $dn) {
					if($dn === $app_dn) { return TRUE; }
				}
			}
		}
		return FALSE;
	}
    
    /*
     * update the application with passed values
     */
    public function save_application($id, $public, $private, $name, $url, $desc, $poc_name, $poc_email, $poc_phone, $request_id, $direct_api_permission, $admin_api_permission) {	
    	$this->load->model('usersmodel');
		$user_id = $this->encrypt->decode($this->session->userdata('user_id'));
    	$permissions = $this->permissions->get_user_permissions($user_id);
    	
		$app = $this->get_application($id)->result(); //get app data from before change
		$query = $this->db->query('UPDATE application SET private_key = '.$this->db->escape($private).',public_key = '.$this->db->escape($public).',name= '.$this->db->escape($name).',url= '.$this->db->escape($url).',description= '.$this->db->escape($desc).',poc_name= '.$this->db->escape($poc_name).',poc_email= '.$this->db->escape($poc_email).',poc_phone= '.$this->db->escape($poc_phone).',application.app_request_id= '.$this->db->escape($request_id).' WHERE application.id = '.$this->db->escape($id));
		if($query) {
			$ldap_conn = $this->prepare_ldap_conn();
			$ldap_bind = $this->usersmodel->ldap_bind_current_user($ldap_conn);
			if($ldap_bind) {
				if($permissions['API']['admins']) {
					//update permission to direct api
					if(isset($direct_api_permission) && isset($admin_api_permission)) { 
						$current_direct_api_permission = $this->get_direct_api_authorization($id);
						//add permission
						if($direct_api_permission === 'on' && !$current_direct_api_permission) {
							ldap_mod_add($ldap_conn,LDAP_DIRECT_API_PERMISSIONS_GROUP,array('member' => $this->get_dn_from_app_id($id)));
						}
						//remove permission
						else if(!$direct_api_permission && $current_direct_api_permission) {
							ldap_mod_del($ldap_conn,LDAP_DIRECT_API_PERMISSIONS_GROUP,array('member' => $this->get_dn_from_app_id($id)));
						}
						
						//update permission to admin api
						$current_admin_api_permission = $this->get_admin_api_authorization($id);
						//add permission
						if($admin_api_permission === 'on' && !$current_admin_api_permission) {
							ldap_mod_add($ldap_conn,LDAP_ADMIN_API_PERMISSIONS_GROUP,array('member' => $this->get_dn_from_app_id($id)));
						}
						//remove permission
						else if(!$admin_api_permission && $current_admin_api_permission) {
							ldap_mod_del($ldap_conn,LDAP_ADMIN_API_PERMISSIONS_GROUP,array('member' => $this->get_dn_from_app_id($id)));
						}
					}
				}
				//change application name in LDAP
				$modify = FALSE;
				$search = ldap_search($ldap_conn, LDAP_BASE_RDN, '(|(cn='.$app[0]->name.'*)(ou='.$app[0]->name.'))', array('*'));
				$entries = ldap_get_entries($ldap_conn, $search);
				foreach($entries as $entry) {
					if(isset($entry['cn']['count']) && (strpos($entry['cn'][0],$app[0]->name) >= 0)) { 
						//change anywhere the application name is used in a common name
						$modify = ldap_modify($ldap_conn,$entry['dn'],array('cn'=>str_replace($app[0]->name,$name,$entry['cn'][0]))); 
					}
				}
			}
			//if LDAP update failed, undo query for name only
			if(!$modify) {
				$undo_query = $this->db->query('UPDATE application SET name = '.$this->db->escape($app[0]->name).' WHERE id = '.$this->db->escape($id));
			}
			return $modify;
		}
		return FALSE;
    }
    
    /*
     * Create a new application, and give it permissions to access the Direct API
     */
    public function create_application($public, $private, $name, $requestor, $url, $desc, $poc_name, $poc_email, $poc_phone, $request_id){
		$this->load->model('usersmodel');
		//create application in database
		$failed = FALSE;
    	$query = $this->db->query('INSERT INTO application (name, public_key, private_key, url, description, poc_name, poc_email, poc_phone, app_request_id) VALUES ('.$this->db->escape($name).','.$this->db->escape($public).','.$this->db->escape($private).','.$this->db->escape($url).','.$this->db->escape($desc).','.$this->db->escape($poc_name).','.$this->db->escape($poc_email).','.$this->db->escape($poc_phone).','.$this->db->escape($request_id).')');
		//create application in LDAP
		if($query) {
			$app_id = $this->get_application_id_from_public($public);
			$ldap_conn = $this->prepare_ldap_conn();
			$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_ADMIN_USERNAME, LDAP_ANON_ADMIN_PASSWORD); //TO-DO: change bind to current user when permissions are set up
			if($ldap_bind) {
				$app_attributes = array(
					'objectClass' => array('organizationalUnit','top','uidObject'),
					'ou' => $name,
					'uid' => $app_id,
				);
				$requestor_username = $this->usersmodel->get_username_from_id($requestor);
				$admins_attributes = array(
					'objectClass' => 'groupOfNames',
					'cn' => $name.' Administrators',
					'member' => array(LDAP_ANON_ADMIN_USERNAME),
				);
				$users_attributes = array(
					'objectClass' => 'groupOfNames',
					'cn' => $name.' Authorized Users',
					'member' => array(LDAP_ANON_ADMIN_USERNAME),
				);
				if(isset($requestor_username) && strlen($requestor_username) > 0) {
					$admins_attributes['member'][1] = $this->usersmodel->get_dn_from_username($requestor_username); //put the requestor in the admins group
					$users_attributes['member'][1] = $this->usersmodel->get_dn_from_username($requestor_username); //put the requestor in the users group
				}
				
				$create_app_in_ldap = ldap_add($ldap_conn, $this->get_dn_from_app_id($app_id), $app_attributes);
				if($create_app_in_ldap) {
					$create_admins_group = ldap_add($ldap_conn, $this->get_admins_dn_from_app_id($app_id), $admins_attributes);
					$create_users_group = ldap_add($ldap_conn, $this->get_users_dn_from_app_id($app_id), $users_attributes);
					$give_api_access = ldap_mod_add($ldap_conn,LDAP_DIRECT_API_PERMISSIONS_GROUP,array('member' => $this->get_dn_from_app_id($app_id)));
					//when a problem comes along, you must whip it... and set this boolean to true
					if(!$create_admins_group || !$create_users_group || !$give_api_access) { $failed = TRUE; }
					else { return TRUE; } //if no problems, we succeeded
				}
				else { $failed = TRUE; }
			}
			//if we failed to create the application in LDAP for any reason, we need to get rid of the bad data we have created
			if($failed) {
				//get rid of the app in LDAP
				$remove_api_access = ldap_mod_del($ldap_conn,LDAP_DIRECT_API_PERMISSIONS_GROUP,array('member' => $this->get_dn_from_app_id($app_id)));
				$delete_admins_group = ldap_delete($ldap_conn, $this->get_admins_dn_from_app_id($app_id));
				$delete_users_group = ldap_delete($ldap_conn, $this->get_users_dn_from_app_id($app_id));
				$delete_app_in_ldap = ldap_delete($ldap_conn, $this->get_dn_from_app_id($app_id));
				//get rid of the app in database
				$this->db->query('DELETE FROM application WHERE id='.$this->db->escape($app_id));
			}
		}
		return FALSE;
    }
	
	/* Simple function to format a dn for a given application on this system.
     * It looks up the application name in the database given the application id to create the dn.
	 */ 
	public function get_dn_from_app_id($app_id) {
		$this->load->model('applicationmodel');
		$app = $this->applicationmodel->get_application($app_id);
		if($app && $app->num_rows() === 1) {
			$row = $app->row_array(0);
			return 'ou='.$row['name'].','.LDAP_APPLICATION_GROUP;
		}
		return FALSE;
	}
	
	/* Simple function to format a dn for a users list for a given application on this system
	 * abstracted to a function so that it can be changed in only one place if the format ever changes.
	 */ 
	public function get_users_dn_from_app_id($app_id) {
		return 'ou='.LDAP_APPLICATION_USER_GROUP_NAME.','.$this->get_dn_from_app_id($app_id);
	}
	
	/* Simple function to format a dn for an admins list for a given application on this system
	 * abstracted to a function so that it can be changed in only one place if the format ever changes.
	 */ 
	public function get_admins_dn_from_app_id($app_id) {
		return 'ou='.LDAP_APPLICATION_ADMIN_GROUP_NAME.','.$this->get_dn_from_app_id($app_id);
	}
	
	public function get_ids($name){
		return  $this->db->query('SELECT id FROM application WHERE name LIKE '.$this->db->escape($name));
	}
	
	/* -----------------------------*
	 *  PRIVATE FUNCTIONS           *
	 * -----------------------------*/
	 
	/* This function prepares a connection to LDAP using the configured constants for the application
	  * and the LDAP options required for the connection. Returns FALSE on failure, LDAP connection resource
	  * on success.
	  */
	 private function prepare_ldap_conn() {
		$ldap_conn = ldap_connect(LDAP_HOSTNAME, LDAP_PORT);
		if(!ldap_set_option($ldap_conn, LDAP_OPT_PROTOCOL_VERSION, 3)) { return FALSE; } 
		if(!ldap_set_option($ldap_conn, LDAP_OPT_REFERRALS, 0)) { return FALSE; }
		return $ldap_conn;
	}
}